#include <visa.h>
#include <ansi_c.h>
#include <formatio.h>
#include "c:\WINDOWS\Desktop\ke7001\Ki7001.h"

#define KI7001_Revision     "Rev 1.0, AL, 04/97, CVI 4.0.1" /*  Instrument driver revision */
#define BUFFER_SIZE         1024L         /* File I/O buffer size */
    
/*= <KEITHLEY 7001> =========================================================*/
/* LabWindows/CVI Instrument Driver                                      	 */
/* Original Release:                                                         */
/* By: Alain Laine                                                           */
/*     PH. 33/1/60/11/51/55   												 */
/*     Fax : 33/1/60/11/77/26     											 */
/*     Email : Laine_alain@keithley.com        				                 */
/*                                                                           */
/* Modification History: None                                                */
/*===========================================================================*/
 
/*****************************************************************************/
/*= INSTRUMENT-DEPENDENT STATUS/RANGE STRUCTURE  ============================*/
/*****************************************************************************/
/* Ki7001_stringValPair is used in the Ki7001_errorMessage function          */
/* Ki7001_statusDataRanges is used to track session dependent status & ranges*/
/*===========================================================================*/
typedef struct  Ki7001_stringValPair
{
   ViStatus stringVal;
   ViString stringName;
}  Ki7001_tStringValPair;
 
    /*=CHANGE:=============================================================*/
    /* Change to reflect the global status variables that your driver needs*/
    /* to keep track of.  For example trigger mode of each session         */
    /*=====================================================================*/
struct Ki7001_statusDataRanges {
    ViInt16 triggerMode;
    ViInt16 val2;
    ViInt16 val3;
    ViChar instrDriverRevision[256];
};

typedef struct Ki7001_statusDataRanges *Ki7001_instrRange;

/*****************************************************************************/
/*= UTILITY ROUTINE DECLARATIONS (Non-Exportable Functions) =================*/
/*****************************************************************************/
ViBoolean Ki7001_invalidViBooleanRange (ViBoolean val);
ViBoolean Ki7001_invalidViInt16Range (ViInt16 val, ViInt16 min, ViInt16 max);
ViBoolean Ki7001_invalidViInt32Range (ViInt32 val, ViInt32 min, ViInt32 max);
ViBoolean Ki7001_invalidViUInt8Range (ViUInt8 val, ViUInt8 min, ViUInt8 max);
ViBoolean Ki7001_invalidViUInt16Range (ViUInt16 val, ViUInt16 min, ViUInt16 max);
ViBoolean Ki7001_invalidViUInt32Range (ViUInt32 val, ViUInt32 min, ViUInt32 max);
ViBoolean Ki7001_invalidViReal32Range (ViReal32 val, ViReal32 min, ViReal32 max);
ViBoolean Ki7001_invalidViReal64Range (ViReal64 val, ViReal64 min, ViReal64 max);
ViStatus Ki7001_initCleanUp (ViSession openRMSession, ViPSession openInstrSession, ViStatus currentStatus);

/*****************************************************************************/
/*------ INSERT INSTRUMENT-DEPENDENT UTILITY ROUTINE DECLARATIONS HERE ------*/
/*****************************************************************************/
ViStatus Ki7001_FrontPanel (ViSession instrumentHandle, ViInt16 window,
                            ViInt16 displayMode, ViChar userMessage[]);
ViStatus Ki7001_TTLOutput (ViSession instrumentHandle, ViInt16 value);
ViStatus Ki7001_TTLInput (ViSession instrumentHandle, ViInt16 *result, int origin);
ViStatus Ki7001_SaveSetup (ViSession instrumentHandle, ViInt16 setupNumber);
ViStatus Ki7001_RecallSetup (ViSession instrumentHandle, ViInt16 setupNumber);
ViStatus Ki7001_POSetup (ViSession instrumentHandle, ViInt16 setup);
ViStatus Ki7001_WriteRegister (ViSession instrumentHandle, ViInt16 reg,
                               ViInt16 value);
ViStatus Ki7001_ReadRegister (ViSession instrumentHandle, ViInt16 reg,
                              ViInt16 *value);
ViStatus Ki7001_ClearRegisters (ViSession instrumentHandle);
ViStatus Ki7001_SetupSRQ (ViSession instrumentHandle, ViInt32 condition);
ViStatus Ki7001_SetupArmLayer (ViSession instrumentHandle, ViInt16 source,
                               ViInt16 count, ViInt16 direction,
                               ViInt16 triggerLinkInputLine,
                               ViInt16 triggerLinkOutputLine);
ViStatus Ki7001_SetupScanLayer (ViSession instrumentHandle, ViInt16 source,
                                ViInt16 count, ViInt16 direction, ViReal64 timer,
                                ViReal64 delay, ViInt16 triggerLinkInputLine,
                                ViInt16 triggerLinkOutputLine);
ViStatus Ki7001_SetupChanLayer (ViSession instrumentHandle, ViInt16 source,
                                ViInt16 count, ViInt16 direction, ViReal64 timer,
                                ViReal64 delay, ViInt16 triggerLinkInputLine,
                                ViInt16 triggerLinkOutputLine);
ViStatus Ki7001_Bypass (ViSession instrumentHandle,  ViInt16 Layer, int bypassDelay);
ViStatus Ki7001_Start (ViSession instrumentHandle, int continuousInit);
ViStatus Ki7001_Stop (ViSession instrumentHandle);
ViStatus Ki7001_DefineScannerCard (ViSession instrumentHandle, ViInt16 slot,
                                   ViInt16 type);
ViStatus Ki7001_QueryScannerCard (ViSession instrumentHandle, ViInt16 slot,
                                  ViInt16 *type);
ViStatus Ki7001_SetPoles (ViSession instrumentHandle, ViInt16 slot, ViInt16 poles);
ViStatus Ki7001_QueryPoles (ViSession instrumentHandle, ViInt16 slot,
                            ViInt16 *poles);
ViStatus Ki7001_SetSettlingTime (ViSession instrumentHandle, ViInt16 slot,
                                 ViReal64 settlingTime);
ViStatus Ki7001_QuerySettlingTime (ViSession instrumentHandle, ViInt16 slot,
                                   ViReal64 *settlingTime);
ViStatus Ki7001_EnableBBM (ViSession instrumentHandle, ViInt16 enable);
ViStatus Ki7001_QueryBBM (ViSession instrumentHandle, ViInt16 *BBM);
ViStatus Ki7001_EnableSingleChan (ViSession instrumentHandle, ViInt16 enable);
ViStatus Ki7001_QuerySingleChan (ViSession instrumentHandle, ViInt16 *mode);
ViStatus Ki7001_EnableCardPairing (ViSession instrumentHandle, ViInt16 enable);
ViStatus Ki7001_QueryCardPairing (ViSession instrumentHandle, ViInt16 *mode);

char *Ki7001_MAKESCANCHAN (char *buffer, int slot, int row, int column);
char *Ki7001_MAKEMUXCHAN (char *buffer, int slot, int channel);
char *Ki7001_MAKEMEMCHAN (char *buffer, int memory);
char *Ki7001_MAKECONTIGUOUSLIST (char *buffer, char *start, char *stop);
char *Ki7001_ADDTOLIST (char *buffer, char *actual, char *new);

ViStatus Ki7001_Close (ViSession instrumentHandle, ViChar channelList[]);
ViStatus Ki7001_Open (ViSession instrumentHandle, ViChar channelList[]);
ViStatus Ki7001_OpenAll (ViSession instrumentHandle);
ViStatus Ki7001_ScanList (ViSession instrumentHandle, ViChar channelList[]);
ViStatus Ki7001_StoreRelayPattern (ViSession instrumentHandle, ViInt16 pattern);
ViStatus Ki7001_RecallRelayPattern (ViSession instrumentHandle, ViInt16 pattern);

/*****************************************************************************/
/*====== USER-CALLABLE FUNCTIONS (Exportable Functions) =====================*/
/*****************************************************************************/

/*===========================================================================*/
/* Function: Initialize                                                      */
/* Purpose:  This function opens the instrument, queries the instrument      */
/*           for its ID, and initializes the instrument to a known state.    */
/*===========================================================================*/
ViStatus _VI_FUNC Ki7001_init (ViRsrc resourceName, 
                    ViBoolean resetDevice, ViPSession instrSession)
{
    ViStatus Ki7001_status = VI_SUCCESS;
    ViSession rmSession = 0;
    ViUInt32 retCnt = 0;
    ViByte rdBuffer[BUFFER_SIZE];

    /*- Check input parameter ranges ----------------------------------------*/
    if (Ki7001_invalidViBooleanRange (resetDevice))
        return VI_ERROR_PARAMETER3;

    /*- Open instrument session ---------------------------------------------*/
    if ((Ki7001_status = viOpenDefaultRM (&rmSession)) < 0)
        return Ki7001_status;
   
    if ((Ki7001_status = viOpen (rmSession, resourceName, VI_NULL, VI_NULL, instrSession)) < 0) {
        viClose (rmSession);
        return Ki7001_status;
    }

    /*- Configure VISA Formatted I/O ----------------------------------------*/
    if ((Ki7001_status = viSetAttribute (*instrSession, VI_ATTR_TMO_VALUE, 10000)) < 0)
            return Ki7001_initCleanUp (rmSession, instrSession, Ki7001_status);
  
    if ((Ki7001_status = viSetBuf (*instrSession, VI_READ_BUF|VI_WRITE_BUF, 4000)) < 0)
            return Ki7001_initCleanUp (rmSession, instrSession, Ki7001_status);
  
    if ((Ki7001_status = viSetAttribute (*instrSession, VI_ATTR_WR_BUF_OPER_MODE,
                            VI_FLUSH_ON_ACCESS)) < 0)
            return Ki7001_initCleanUp (rmSession, instrSession, Ki7001_status);
    
    if ((Ki7001_status = viSetAttribute (*instrSession, VI_ATTR_RD_BUF_OPER_MODE,
                            VI_FLUSH_ON_ACCESS)) < 0)
            return Ki7001_initCleanUp (rmSession, instrSession, Ki7001_status);
 
    /*- Reset instrument ----------------------------------------------------*/
    if (resetDevice) {
        if ((Ki7001_status = Ki7001_reset (*instrSession)) < 0)
            return Ki7001_initCleanUp (rmSession, instrSession, Ki7001_status);
    }       
          
    return Ki7001_status;
}

/*===========================================================================*/
/* Function: Reset                                                           */
/* Purpose:  This function resets the instrument.  If the reset function     */
/*           is not supported by the instrument, this function returns       */
/*           the warning VI_WARN_NSUP_RESET.                                 */
/*===========================================================================*/
ViStatus _VI_FUNC Ki7001_reset (ViSession instrSession)
{
    ViUInt32 retCnt = 0;
    ViStatus Ki7001_status = VI_SUCCESS;

    /*  Initialize the instrument to a known state.  */
    if ((Ki7001_status = viWrite (instrSession, "*RST", 4, &retCnt)) < 0)
        return Ki7001_status;

       
    return Ki7001_status;
}

/*===========================================================================*/
/* Function: Self-Test                                                       */
/* Purpose:  This function executes the instrument self-test and returns     */
/*           the result. If the self test function is not supported by the   */
/*           instrument, this function returns the warning                   */
/*           VI_WARN_NSUP_SELF_TEST.                                         */
/*===========================================================================*/
ViStatus _VI_FUNC Ki7001_selfTest (ViSession instrSession, ViPInt16 testResult,
                    ViChar _VI_FAR testMessage[])
{
    ViUInt32 retCnt = 0;
    ViStatus Ki7001_status = VI_SUCCESS;

    /*=CHANGE:=============================================================*/
    /* Change the *TST? command to the self-test command for your device.  */
    /*=====================================================================*/
    if ((Ki7001_status = viWrite (instrSession, "*TST?", 5, &retCnt)) < 0)
        return Ki7001_status;

    /*=CHANGE:=============================================================*/
    /* Test the test_message to see if a self-test error occurred. Assign  */
    /* the result to *test_result and return. Zero (0) means that the test */
    /* executed successfully any other number means selftest failed see    */
    /* control help for more details.                                      */
    /* This code makes the following assumption: The String returned from  */
    /* the instrument is in the format set forth in the SCPI 1994 Command  */
    /* Reference                                                           */
    /*=====================================================================*/
 
    if ((Ki7001_status = viScanf (instrSession, "%d", testResult)) < 0)
    
    testMessage[0]=0;
    
        return Ki7001_status;
    
    /*=CHANGE: ==============================================================*/
    /* If the Instrument Does Not support a Self Test, delete the above      */
    /* section that sends the Self Test command to the instrument and parses */
    /* the response and use the following line of code to return the proper  */ 
    /* warning status code.                                                  */
    /*                                                                       */
    /* Ki7001_status = VI_WARN_NSUP_SELF_TEST;                               */
    /*                                                                       */ 
    /* If a Self Test is supported delete this section of comment and code.  */
    /*=======================================================================*/

    return Ki7001_status;
}

/*===========================================================================*/
/* Function: Error Query                                                     */
/* Purpose:  This function queries the instrument error queue, and returns   */
/*           the result. If the error query function is not supported by the  */
/*           instrument, this function returns the warning                   */
/*           VI_WARN_NSUP_ERROR_QUERY.                                       */
/*===========================================================================*/
ViStatus _VI_FUNC Ki7001_errorQuery (ViSession instrSession, ViPInt32 errorCode,
                    ViChar _VI_FAR errorMessage[])
{
    ViUInt32 retCnt = 0;
    ViStatus Ki7001_status = VI_SUCCESS;

    /*=CHANGE:=============================================================*/
    /* Change the :SYST:ERR? command to the Error Query command for your   */
    /* device.                                                             */
    /*=====================================================================*/
    if ((Ki7001_status = viWrite (instrSession, ":SYST:ERR?", 10, &retCnt)) < 0)
        return Ki7001_status;

    /*=CHANGE:=============================================================*/
    /* Parse the error message and return the Error Code and Error String. */
    /* This code makes the following assumption: The String returned from  */
    /* the instrument is in the format set forth in the SCPI 1994 Command  */
    /* Reference which is <Error/Event Number,"Error/Event Descriptor">,   */
    /* for example  0,"No Error" or -410,"Query Interrupted".              */
    /*=====================================================================*/

    if ((Ki7001_status = viScanf (instrSession, "%ld,\"%[^\"]", errorCode, errorMessage)) < 0)
        return Ki7001_status;

    /*=CHANGE: ==============================================================*/
    /* If the Instrument Does Not support an Error Query, delete the above   */
    /* section that sends the Error Query command to the instrument and      */
    /* parses the response and use the following line of code to return the  */ 
    /* proper warning status code.                                           */
    /*                                                                       */
    /* Ki7001_status = VI_WARN_NSUP_ERROR_QUERY;                             */
    /*                                                                       */ 
    /* If a Error Query is supported delete this section of comment and code.*/                                          
    /*=======================================================================*/

    return Ki7001_status;
}

/*===========================================================================*/
/* Function: Error Message                                                   */
/* Purpose:  This function translates the error return value from the        */
/*           instrument driver into a user-readable string.                  */
/*===========================================================================*/
ViStatus _VI_FUNC Ki7001_errorMessage (ViSession instrSession, ViStatus statusCode,
                    ViChar _VI_FAR message[])
{
    ViStatus Ki7001_status = VI_SUCCESS;
    ViInt16 i;
    static Ki7001_tStringValPair statusDescArray[] = {
        {VI_WARN_NSUP_ID_QUERY,     "WARNING: ID Query not supported"},
        {VI_WARN_NSUP_RESET,        "WARNING: Reset not supported"},
        {VI_WARN_NSUP_SELF_TEST,    "WARNING: Self-test not supported"},
        {VI_WARN_NSUP_ERROR_QUERY,  "WARNING: Error Query not supported"},     
        {VI_WARN_NSUP_REV_QUERY,    "WARNING: Revision Query not supported"},
        {VI_ERROR_PARAMETER1,   "ERROR: Parameter 1 out of range"},
        {VI_ERROR_PARAMETER2,   "ERROR: Parameter 2 out of range"},
        {VI_ERROR_PARAMETER3,   "ERROR: Parameter 3 out of range"},
        {VI_ERROR_PARAMETER4,   "ERROR: Parameter 4 out of range"},
        {VI_ERROR_PARAMETER5,   "ERROR: Parameter 5 out of range"},
        {VI_ERROR_PARAMETER6,   "ERROR: Parameter 6 out of range"},
        {VI_ERROR_PARAMETER7,   "ERROR: Parameter 7 out of range"},
        {VI_ERROR_PARAMETER8,   "ERROR: Parameter 8 out of range"},
        {VI_ERROR_FAIL_ID_QUERY,"ERROR: Identification query failed"},
        {VI_ERROR_INV_RESPONSE, "ERROR: Interpreting instrument response"},
        {VI_ERROR_INSTR_FILE_OPEN,    "ERROR: Opening the specified file"},
        {VI_ERROR_INSTR_FILE_WRITE,   "ERROR: Writing to the specified file"},
        {VI_ERROR_INSTR_INTERPRETING_RESPONSE, "ERROR: Interpreting the instrument's response"},
        {Ki7001_ERROR_INVALID_CONFIGURATION, "Error: Instrument reported an error, use Ki2304_errorQuery"},
        
        {VI_NULL, VI_NULL}
    };

    Ki7001_status = viStatusDesc (instrSession, statusCode, message);
    if (Ki7001_status == VI_WARN_UNKNOWN_STATUS) {
        for (i=0; statusDescArray[i].stringName; i++) {
            if (statusDescArray[i].stringVal == statusCode) {
                strcpy (message, statusDescArray[i].stringName);
                return (VI_SUCCESS);
            }
        }
        sprintf (message, "Unknown Error 0x%08lX", statusCode);
        return (VI_WARN_UNKNOWN_STATUS);
    }
    
    Ki7001_status = VI_SUCCESS;
    return Ki7001_status;
}

/*===========================================================================*/
/* Function: Revision Query                                                  */
/* Purpose:  This function returns the driver and instrument revisions.      */
/*           If the revision query function is not supported by the          */ 
/*           instrument, this function returns the warning                   */
/*           VI_WARN_NSUP_REV_QUERY.                                         */
/*===========================================================================*/
ViStatus _VI_FUNC Ki7001_revisionQuery (ViSession instrSession,
                    ViChar _VI_FAR driverRev[], ViChar _VI_FAR instrRev[])
{
    ViUInt32 retCnt = 0;
    ViStatus Ki7001_status = VI_SUCCESS;

    /*=CHANGE:=============================================================*/
    /* Query instrument firmware revision.                                 */
    /* Change the ID Query command to the ID Query command for your device.*/
    /*=====================================================================*/
    if ((Ki7001_status = viWrite (instrSession, "*IDN?", 5, &retCnt)) < 0)
        return Ki7001_status;

    /*=CHANGE:=============================================================*/
    /* This code makes the following assumption: The String returned from  */
    /* the instrument is in the format set forth in the SCPI 1994 Command  */
    /* Reference                                                           */
    /*=====================================================================*/
    if ((Ki7001_status = viScanf (instrSession, "%*[^,],%*[^,],%*[^,],%[^\n]", instrRev)) < 0)
        return Ki7001_status;

    /*=CHANGE: ==============================================================*/
    /* If the Instrument Does Not support a Revision Query, delete the above */
    /* section that sends the Revision Query command to the instrument and   */
    /* parses the response and use the following line of code to return the  */ 
    /* proper warning status code.                                           */
    /*                                                                       */
    /* Ki7001_status = VI_WARN_NSUP_REV_QUERY;                               */
    /*                                                                       */
    /* If a Revision Query is supported delete this section of comment and   */
    /* code.                                                                 */
    /*=======================================================================*/
    
    strcpy (driverRev, KI7001_Revision);
    
    return Ki7001_status;
}

/*===========================================================================*/
/* Function: Close                                                           */
/* Purpose:  This function closes the instrument.                            */
/*===========================================================================*/
ViStatus _VI_FUNC Ki7001_close (ViSession instrSession)
{
    Ki7001_instrRange instrPtr;
    ViSession rmSession;
    ViStatus Ki7001_status = VI_SUCCESS;

    if ((Ki7001_status = viGetAttribute (instrSession, VI_ATTR_RM_SESSION, &rmSession)) < 0)
        return Ki7001_status;
    if ((Ki7001_status = viGetAttribute (instrSession, VI_ATTR_USER_DATA, &instrPtr)) < 0)
        return Ki7001_status;
            
    if (instrPtr != NULL) 
        free (instrPtr);
    
    Ki7001_status = viClose (instrSession);
    viClose (rmSession);

    return Ki7001_status;
}

/*****************************************************************************/
/*= UTILITY ROUTINES (Non-Exportable Functions) =============================*/
/*****************************************************************************/

/*===========================================================================*/
/* Function: Boolean Value Out Of Range - ViBoolean                          */
/* Purpose:  This function checks a Boolean to see if it is equal to VI_TRUE */
/*           or VI_FALSE. If the value is out of range, the return value is  */
/*           VI_TRUE, otherwise the return value is VI_FALSE.                */
/*===========================================================================*/
ViBoolean Ki7001_invalidViBooleanRange (ViBoolean val)
{
    return ((val != VI_FALSE && val != VI_TRUE) ? VI_TRUE : VI_FALSE);
}

/*===========================================================================*/
/* Function: Short Signed Integer Value Out Of Range - ViInt16               */
/* Purpose:  This function checks a short signed integer value to see if it  */  
/*           lies between a minimum and maximum value.  If the value is out  */
/*           of range, the return value is VI_TRUE, otherwise the return     */
/*           value is VI_FALSE.                                              */
/*===========================================================================*/
ViBoolean Ki7001_invalidViInt16Range (ViInt16 val, ViInt16 min, ViInt16 max)
{
    return ((val < min || val > max) ? VI_TRUE : VI_FALSE);
}

/*===========================================================================*/
/* Function: Long Signed Integer Value Out Of Range - ViInt32                */
/* Purpose:  This function checks a long signed integer value to see if it   */  
/*           lies between a minimum and maximum value.  If the value is out  */
/*           of range, the return value is VI_TRUE, otherwise the return     */
/*           value is VI_FALSE.                                              */
/*===========================================================================*/
ViBoolean Ki7001_invalidViInt32Range (ViInt32 val, ViInt32 min, ViInt32 max)
{
    return ((val < min || val > max) ? VI_TRUE : VI_FALSE);
}

/*===========================================================================*/
/* Function: Unsigned Char Value Out Of Range - ViUInt8                      */
/* Purpose:  This function checks an unsigned char value to see if it        */  
/*           lies between a minimum and maximum value.  If the value is out  */
/*           of range, the return value is VI_TRUE, otherwise the return     */
/*           value is VI_FALSE.                                              */
/*===========================================================================*/
ViBoolean Ki7001_invalidViUInt8Range (ViUInt8 val, ViUInt8 min, ViUInt8 max)
{
    return ((val < min || val > max) ? VI_TRUE : VI_FALSE);
}

/*===========================================================================*/
/* Function: Short Unsigned Integer Value Out Of Range - ViUInt16            */
/* Purpose:  This function checks a short unsigned integer value to see if it*/  
/*           lies between a minimum and maximum value.  If the value is out  */
/*           of range, the return value is VI_TRUE, otherwise the return     */
/*           value is VI_FALSE.                                              */
/*===========================================================================*/
ViBoolean Ki7001_invalidViUInt16Range (ViUInt16 val, ViUInt16 min, ViUInt16 max)
{
    return ((val < min || val > max) ? VI_TRUE : VI_FALSE);
}

/*===========================================================================*/
/* Function: Long Unsigned Integer Value Out Of Range - ViUInt32             */
/* Purpose:  This function checks a long unsigned integer value to see if it */  
/*           lies between a minimum and maximum value.  If the value is out  */
/*           of range, the return value is VI_TRUE, otherwise the return     */
/*           value is VI_FALSE.                                              */
/*===========================================================================*/
ViBoolean Ki7001_invalidViUInt32Range (ViUInt32 val, ViUInt32 min, ViUInt32 max)
{
    return ((val < min || val > max) ? VI_TRUE : VI_FALSE);
}

/*===========================================================================*/
/* Function: Real (Float) Value Out Of Range - ViReal32                      */
/* Purpose:  This function checks a real (float) value to see if it lies     */  
/*           between a minimum and maximum value.  If the value is out of    */
/*           range, the return value is VI_TRUE, otherwise the return value  */
/*           is VI_FALSE.                                                    */
/*===========================================================================*/
ViBoolean Ki7001_invalidViReal32Range (ViReal32 val, ViReal32 min, ViReal32 max)
{
    return ((val < min || val > max) ? VI_TRUE : VI_FALSE);
}

/*===========================================================================*/
/* Function: Real (Double) Value Out Of Range - ViReal64                     */
/* Purpose:  This function checks a real (double) value to see if it lies    */  
/*           between a minimum and maximum value.  If the value is out of    */
/*           range, the return value is VI_TRUE, otherwise the return value  */
/*           is VI_FALSE.                                                    */
/*===========================================================================*/
ViBoolean Ki7001_invalidViReal64Range (ViReal64 val, ViReal64 min, ViReal64 max)
{
    return ((val < min || val > max) ? VI_TRUE : VI_FALSE);
}

/*===========================================================================*/
/* Function: Initialize Clean Up                                             */
/* Purpose:  This function is used only by the Ki7001_init function.  When   */
/*           an error is detected this function is called to close the       */
/*           open resource manager and instrument object sessions and to     */
/*           set the instrSession that is returned from Ki7001_init to       */
/*           VI_NULL.                                                        */
/*===========================================================================*/
ViStatus Ki7001_initCleanUp (ViSession openRMSession,
                    ViPSession openInstrSession, ViStatus currentStatus)
{
    viClose (*openInstrSession);
    viClose (openRMSession);
    *openInstrSession = VI_NULL;
    
    return currentStatus;
}

/*****************************************************************************/
/*----------- INSERT INSTRUMENT-DEPENDENT UTILITY ROUTINES HERE -------------*/
/*****************************************************************************/

ViStatus Ki7001_FrontPanel (ViSession instrumentHandle, ViInt16 window,
                            ViInt16 displayMode, ViChar userMessage[])
{
	char buf[256];
    ViUInt32 retCnt = 0;
    ViInt32 errorCode;
    ViChar errorMessage[256];
    ViStatus Ki7001_status = VI_SUCCESS;
    
    if(displayMode==KI7001_USER_DISPLAY)
    {
		Fmt(buf,"%s<%s%d%s%s%s",":DISP:WIND",window,":TEXT:STAT ON; DATA '",userMessage,"'");
    }
    else
    {
		Fmt(buf,"%s<%s%d%s",":DISP:WIND",window,":TEXT:STAT OFF");
    }
    
    if ((Ki7001_status = viWrite (instrumentHandle, buf, StringLength(buf), &retCnt)) < 0)
        return Ki7001_status;

	if ((Ki7001_status = Ki7001_errorQuery (instrumentHandle, &errorCode, errorMessage)) < 0)
		return Ki7001_status;

	if(errorCode!=0) 
		return Ki7001_ERROR_INVALID_CONFIGURATION;
	
	return Ki7001_status;
}

ViStatus Ki7001_TTLOutput (ViSession instrumentHandle, ViInt16 value)
{
	char buf[256];
    ViUInt32 retCnt = 0;
    ViInt32 errorCode;
    ViChar errorMessage[256];
    ViStatus Ki7001_status = VI_SUCCESS;
    
    if(value & 0x01)
    {
		Fmt(buf,"%s<%s",":SOUR:TTL1:LEV 1; ");
    }
    else
	{
		Fmt(buf,"%s<%s",":SOUR:TTL1:LEV 0; ");
	}
	
    if(value & 0x02)
    {
		Fmt(buf,"%s<%s%s",buf,":SOUR:TTL2:LEV 1; ");
    }
    else
	{
		Fmt(buf,"%s<%s%s",buf,":SOUR:TTL2:LEV 0; ");
	}

    if(value & 0x04)
    {
		Fmt(buf,"%s<%s%s",buf,":SOUR:TTL3:LEV 1; ");
    }
    else
	{
		Fmt(buf,"%s<%s%s",buf,":SOUR:TTL3:LEV 0; ");
	}

    if(value & 0x08)
    {
		Fmt(buf,"%s<%s%s",buf,":SOUR:TTL4:LEV 1;");
    }
    else
	{
		Fmt(buf,"%s<%s%s",buf,":SOUR:TTL4:LEV 0;");
	}

    if ((Ki7001_status = viWrite (instrumentHandle, buf, StringLength(buf), &retCnt)) < 0)
        return Ki7001_status;

	if ((Ki7001_status = Ki7001_errorQuery (instrumentHandle, &errorCode, errorMessage)) < 0)
		return Ki7001_status;

	if(errorCode!=0) 
		return Ki7001_ERROR_INVALID_CONFIGURATION;
	
	return Ki7001_status;
}

ViStatus Ki7001_TTLInput (ViSession instrumentHandle, ViInt16 *result, int origin)
{
	char buf[256];
    ViUInt32 retCnt = 0;
    ViInt32 errorCode;
    ViChar errorMessage[256];
    ViStatus Ki7001_status = VI_SUCCESS;
    
    if(origin==KI7001_INTERNAL_TTL)
		Fmt(buf,"%s<%s",":SENS1:TTL1:DATA?");
	else
	{
		Fmt(buf,"%s<%s%d%s",":SENS",origin,":DATA?");
	}
    
    if ((Ki7001_status = viWrite (instrumentHandle, buf, StringLength(buf), &retCnt)) < 0)
        return Ki7001_status;

    if ((Ki7001_status = viScanf (instrumentHandle, "%d", result)) < 0)
        return Ki7001_status;

	if ((Ki7001_status = Ki7001_errorQuery (instrumentHandle, &errorCode, errorMessage)) < 0)
		return Ki7001_status;

	if(errorCode!=0) 
		return Ki7001_ERROR_INVALID_CONFIGURATION;
	
	return Ki7001_status;
}

ViStatus Ki7001_SaveSetup (ViSession instrumentHandle, ViInt16 setupNumber)
{
	char buf[256];
    ViUInt32 retCnt = 0;
    ViInt32 errorCode;
    ViChar errorMessage[256];
    ViStatus Ki7001_status = VI_SUCCESS;
    
	Fmt(buf,"%s<%s%d","*SAV ",setupNumber);
    
    if ((Ki7001_status = viWrite (instrumentHandle, buf, StringLength(buf), &retCnt)) < 0)
        return Ki7001_status;

	if ((Ki7001_status = Ki7001_errorQuery (instrumentHandle, &errorCode, errorMessage)) < 0)
		return Ki7001_status;

	if(errorCode!=0) 
		return Ki7001_ERROR_INVALID_CONFIGURATION;
	
	return Ki7001_status;
}

ViStatus Ki7001_RecallSetup (ViSession instrumentHandle, ViInt16 setupNumber)
{
	char buf[256];
    ViUInt32 retCnt = 0;
    ViInt32 errorCode;
    ViChar errorMessage[256];
    ViStatus Ki7001_status = VI_SUCCESS;
    
	Fmt(buf,"%s<%s%d","*RCL ",setupNumber);
    
    if ((Ki7001_status = viWrite (instrumentHandle, buf, StringLength(buf), &retCnt)) < 0)
        return Ki7001_status;

	if ((Ki7001_status = Ki7001_errorQuery (instrumentHandle, &errorCode, errorMessage)) < 0)
		return Ki7001_status;

	if(errorCode!=0) 
		return Ki7001_ERROR_INVALID_CONFIGURATION;
	
	return Ki7001_status;
}

ViStatus Ki7001_POSetup (ViSession instrumentHandle, ViInt16 setup)
{
	char buf[256];
    ViUInt32 retCnt = 0;
    ViInt32 errorCode;
    ViChar errorMessage[256];
    ViStatus Ki7001_status = VI_SUCCESS;

    switch(setup)
    {
	case PRES_SETUP:				
		Fmt(buf,"%s<%s",":SYST:POS PRES");
		break;
	case USER0_SETUP:					
		Fmt(buf,"%s<%s",":SYST:POS SAV0");
		break;
	case USER1_SETUP:					
		Fmt(buf,"%s<%s",":SYST:POS SAV1");
		break;
	case USER2_SETUP:					
		Fmt(buf,"%s<%s",":SYST:POS SAV2");
		break;
	case USER3_SETUP:					
		Fmt(buf,"%s<%s",":SYST:POS SAV3");
		break;
	case USER4_SETUP:					
		Fmt(buf,"%s<%s",":SYST:POS SAV4");
		break;
	case USER5_SETUP:					
		Fmt(buf,"%s<%s",":SYST:POS SAV5");
		break;
	case USER6_SETUP:					
		Fmt(buf,"%s<%s",":SYST:POS SAV6");
		break;
	case USER7_SETUP:					
		Fmt(buf,"%s<%s",":SYST:POS SAV7");
		break;
	case USER8_SETUP:					
		Fmt(buf,"%s<%s",":SYST:POS SAV8");
		break;
	case USER9_SETUP:
		Fmt(buf,"%s<%s",":SYST:POS SAV9");
		break;
	default:
		Fmt(buf,"%s<%s",":SYST:POS RST");
    }
    
    if ((Ki7001_status = viWrite (instrumentHandle, buf, StringLength(buf), &retCnt)) < 0)
        return Ki7001_status;

	if ((Ki7001_status = Ki7001_errorQuery (instrumentHandle, &errorCode, errorMessage)) < 0)
		return Ki7001_status;

	if(errorCode!=0) 
		return Ki7001_ERROR_INVALID_CONFIGURATION;
	
	return Ki7001_status;
}

ViStatus Ki7001_WriteRegister (ViSession instrumentHandle, ViInt16 reg,
                               ViInt16 value)
{
	char buf[256];
	char *regName;
    ViUInt32 retCnt = 0;
    ViInt32 errorCode;
    ViChar errorMessage[256];
    ViStatus Ki7001_status = VI_SUCCESS;
    
    switch(reg)
    {
	case KI7001_REG_SRE:
    	regName="*SRE ";
    	break;
	case KI7001_REG_ESE:
    	regName="*ESE ";
    	break;
	case KI7001_REG_OPER_ENAB:
    	regName=":STAT:OPER:ENAB ";
    	break;
	case KI7001_REG_OPER_TRANS:
    	regName=":STAT:OPER:PTR ";
    	break;
	case KI7001_REG_ARM_ENAB:
    	regName=":STAT:OPER:ARM:ENAB ";
    	break;
	case KI7001_REG_ARM_TRANS:
    	regName=":STAT:OPER:ARM:PTR ";
    	break;
	case KI7001_REG_SEQ_ENAB:
    	regName=":STAT:OPER:ARM:SEQ:ENAB ";
    	break;
	case KI7001_REG_SEQ_TRANS:
    	regName=":STAT:OPER:ARM:SEQ:PTR ";
    	break;
	case KI7001_REG_QUES_ENAB:
    	regName=":STAT:QUES:ENAB ";
    	break;
	case KI7001_REG_QUES_TRANS:
    	regName=":STAT:QUES:PTR ";
    	break;
	case KI7001_REG_TRIG_ENAB:
    	regName=":STAT:OPER:TRIG:ENAB ";
    	break;
	case KI7001_REG_TRIG_TRANS:
    	regName=":STAT:OPER:TRIG:PTR ";
    	break;
	default:
		return VI_ERROR_PARAMETER2;
	}

	Fmt(buf,"%s<%s%d",regName,value);
    if ((Ki7001_status = viWrite (instrumentHandle, buf, StringLength(buf), &retCnt)) < 0)
        return Ki7001_status;

	if ((Ki7001_status = Ki7001_errorQuery (instrumentHandle, &errorCode, errorMessage)) < 0)
		return Ki7001_status;
	if(errorCode!=0) 
		return Ki7001_ERROR_INVALID_CONFIGURATION;
	
	return Ki7001_status;
}

ViStatus Ki7001_ReadRegister (ViSession instrumentHandle, ViInt16 reg,
                              ViInt16 *value)
{
	char buf[256];
	char *regName;
    ViUInt32 retCnt = 0;
    ViInt32 errorCode;
    ViChar errorMessage[256];
    ViStatus Ki7001_status = VI_SUCCESS;
    
    switch(reg)
    {
	case KI7001_REG_SRE:
    	regName="*SRE?";
    	break;
	case KI7001_REG_ESE:
    	regName="*ESE?";
    	break;
	case KI7001_REG_OPER_ENAB:
    	regName=":STAT:OPER:ENAB?";
    	break;
	case KI7001_REG_OPER_TRANS:
    	regName=":STAT:OPER:PTR?";
    	break;
	case KI7001_REG_ARM_ENAB:
    	regName=":STAT:OPER:ARM:ENAB?";
    	break;
	case KI7001_REG_ARM_TRANS:
    	regName=":STAT:OPER:ARM_PTR?";
    	break;
	case KI7001_REG_SEQ_ENAB:
    	regName=":STAT:OPER:ARM:SEQ:ENAB?";
    	break;
	case KI7001_REG_SEQ_TRANS:
    	regName=":STAT:OPER:ARM:SEQ:PTR?";
    	break;
	case KI7001_REG_QUES_ENAB:
    	regName=":STAT:QUES:ENAB?";
    	break;
	case KI7001_REG_QUES_TRANS:
    	regName=":STAT:QUES:PTR?";
    	break;
	case KI7001_REG_TRIG_ENAB:
    	regName=":STAT:OPER:TRIG:ENAB?";
    	break;
	case KI7001_REG_TRIG_TRANS:
    	regName=":STAT:OPER:TRIG:PTR?";
    	break;
	case KI7001_REG_STB:
    	regName="*STB?";
    	break;
	case KI7001_REG_ESE_EVENT:
    	regName="*ESR?";
    	break;
	case KI7001_REG_OPER_EVENT:
    	regName=":STAT:OPER:EVENT?";
    	break;
	case KI7001_REG_OPER_COND:
    	regName=":STAT:OPER:COND?";
    	break;
	case KI7001_REG_ARM_EVENT:
    	regName=":STAT:OPER:ARM:EVENT?";
    	break;
	case KI7001_REG_ARM_COND:
    	regName=":STAT:OPER:ARM:COND?";
    	break;
	case KI7001_REG_SEQ_EVENT:
    	regName=":STAT:OPER:ARM:SEQ:EVENT?";
    	break;
	case KI7001_REG_SEQ_COND:
    	regName=":STAT:OPER:ARM:SEQ:COND?";
    	break;
	case KI7001_REG_QUES_EVENT:
    	regName=":STAT:QUES:EVENT?";
    	break;
	case KI7001_REG_QUES_COND:
    	regName=":STAT:QUEST:COND?";
    	break;
	case KI7001_REG_TRIG_EVENT:
    	regName=":STAT:OPER:TRIG:EVENT?";
    	break;
	case KI7001_REG_TRIG_COND:
    	regName=":STAT:OPER:TRIG:COND?";
    	break;
	default:
		return VI_ERROR_PARAMETER2;
	}

	Fmt(buf,"%s<%s",regName);
    if ((Ki7001_status = viWrite (instrumentHandle, buf, StringLength(buf), &retCnt)) < 0)
        return Ki7001_status;

    if ((Ki7001_status = viScanf (instrumentHandle, "%d", value)) < 0)
        return Ki7001_status;

	if ((Ki7001_status = Ki7001_errorQuery (instrumentHandle, &errorCode, errorMessage)) < 0)
		return Ki7001_status;
	if(errorCode!=0) 
		return Ki7001_ERROR_INVALID_CONFIGURATION;
	
	return Ki7001_status;
}

ViStatus Ki7001_ClearRegisters (ViSession instrumentHandle)
{
	char buf[256];
    ViUInt32 retCnt = 0;
    ViInt32 errorCode;
    ViChar errorMessage[256];
    ViStatus Ki7001_status = VI_SUCCESS;

	Fmt(buf,"%s<%s","*ESE 0; *SRE 0; :STAT:QUES:ENAB 0; :STAT:OPER:ENAB 0; :STAT:OPER:ARM:ENAB 0; :STAT:OPER:ARM:SEQ:ENAB 0; :STAT:OPER:TRIG:ENAB 0");
    if ((Ki7001_status = viWrite (instrumentHandle, buf, StringLength(buf), &retCnt)) < 0)
        return Ki7001_status;

	if ((Ki7001_status = Ki7001_errorQuery (instrumentHandle, &errorCode, errorMessage)) < 0)
		return Ki7001_status;
	if(errorCode!=0) 
		return Ki7001_ERROR_INVALID_CONFIGURATION;
	
	return Ki7001_status;
}

ViStatus Ki7001_SetupSRQ (ViSession instrumentHandle, ViInt32 condition)
{
	char buf[256];
    ViUInt32 retCnt = 0;
    ViInt32 errorCode;
    ViChar errorMessage[256];
    ViStatus Ki7001_status = VI_SUCCESS;

	ViInt16 armReg=0;
	ViInt16 trigReg=0;
	ViInt16 seqReg=0;
	ViInt16 operReg=0;
	ViInt16 stdEvReg=0;
	ViInt16 sreReg=0;
	
	if(condition == 0)
	{
    	Ki7001_status = Ki7001_ClearRegisters (instrumentHandle);
        return Ki7001_status;
    }
    
    // Read the actual registers value first
	if ((Ki7001_status = Ki7001_ReadRegister(instrumentHandle, 0, &sreReg)) < 0)
		return Ki7001_status;
	if ((Ki7001_status = Ki7001_ReadRegister(instrumentHandle, 1, &stdEvReg)) < 0)
		return Ki7001_status;
	if ((Ki7001_status = Ki7001_ReadRegister(instrumentHandle, 2, &operReg)) < 0)
		return Ki7001_status;
	if ((Ki7001_status = Ki7001_ReadRegister(instrumentHandle, 4, &armReg)) < 0)
		return Ki7001_status;
	if ((Ki7001_status = Ki7001_ReadRegister(instrumentHandle, 10, &trigReg)) < 0)
		return Ki7001_status;
	if ((Ki7001_status = Ki7001_ReadRegister(instrumentHandle, 6, &seqReg)) < 0)
		return Ki7001_status;

	if(condition & KI7001_SRQ_ON_IN_TRIG_LAYER)				
	{
		operReg |= 0x0020;
		trigReg |= 0x0002;
		sreReg |= 0x80;
	}
	if(condition & KI7001_SRQ_ON_IN_ARM_LAYER1)				
	{
		operReg |= 0x0040;
		armReg |= 0x0002;
		seqReg |= 0x0002;
		sreReg |= 0x80;
	}
	if(condition & KI7001_SRQ_ON_IN_ARM_LAYER2)				
	{
		operReg |= 0x0040;
		armReg |= 0x0002;
		seqReg |= 0x0004;
		sreReg |= 0x80;
	}
    
	if(condition & KI7001_SRQ_ON_SETTLING)					
	{
		operReg |= 0x0002;
		sreReg |= 0x80;
	}

	if(condition & KI7001_SRQ_ON_IN_IDLE)					
	{
		operReg |= 0x0400;
		sreReg |= 0x80;
	}

	if(condition & KI7001_SRQ_ON_ERROR_AVAIL)				
	{
		sreReg |= 0x04;
	}

	if(condition & KI7001_SRQ_ON_MESSAGE_AVAIL)				
	{
		sreReg |= 0x10;
	}

	if(condition & KI7001_SRQ_ON_OP_COMPLETED)				
	{
		sreReg |= 0x20;
		stdEvReg |= 0x01;
	}

	if(condition & KI7001_SRQ_ON_REQCTRL)					
	{
		sreReg |= 0x20;
		stdEvReg |= 0x02;
	}

	if(condition & KI7001_SRQ_ON_DEVICE_ERROR)				
	{
		sreReg |= 0x20;
		stdEvReg |= 0x04;
	}

	if(condition & KI7001_SRQ_ON_EXEC_ERROR)				
	{
		sreReg |= 0x20;
		stdEvReg |= 0x10;
	}

	if(condition & KI7001_SRQ_ON_CMD_ERROR)					
	{
		sreReg |= 0x20;
		stdEvReg |= 0x20;
	}

	if(condition & KI7001_SRQ_ON_USER_REQUEST)					
	{
		sreReg |= 0x20;
		stdEvReg |= 0x40;
	}

	if ((Ki7001_status = Ki7001_WriteRegister(instrumentHandle, 0, sreReg)) < 0)
		return Ki7001_status;
	if ((Ki7001_status = Ki7001_WriteRegister(instrumentHandle, 1, stdEvReg)) < 0)
		return Ki7001_status;
	if ((Ki7001_status = Ki7001_WriteRegister(instrumentHandle, 2, operReg)) < 0)
		return Ki7001_status;
	if ((Ki7001_status = Ki7001_WriteRegister(instrumentHandle, 4, armReg)) < 0)
		return Ki7001_status;
	if ((Ki7001_status = Ki7001_WriteRegister(instrumentHandle, 6, seqReg)) < 0)
		return Ki7001_status;
	if ((Ki7001_status = Ki7001_WriteRegister(instrumentHandle, 10, trigReg)) < 0)
		return Ki7001_status;
	if ((Ki7001_status = Ki7001_WriteRegister(instrumentHandle, 3, operReg)) < 0)
		return Ki7001_status;
	if ((Ki7001_status = Ki7001_WriteRegister(instrumentHandle, 5, armReg)) < 0)
		return Ki7001_status;
	if ((Ki7001_status = Ki7001_WriteRegister(instrumentHandle, 7, seqReg)) < 0)
		return Ki7001_status;
	if ((Ki7001_status = Ki7001_WriteRegister(instrumentHandle, 11, trigReg)) < 0)
		return Ki7001_status;

	return Ki7001_status;
}

ViStatus Ki7001_SetupArmLayer (ViSession instrumentHandle, ViInt16 source,
                               ViInt16 count, ViInt16 direction,
                               ViInt16 triggerLinkInputLine,
                               ViInt16 triggerLinkOutputLine)
{
	char buf[256];
	char countbuf[25];
	char *szSource;
	char *szDirection;
    ViUInt32 retCnt = 0;
    ViInt32 errorCode;
    ViChar errorMessage[256];
    ViStatus Ki7001_status = VI_SUCCESS;

	switch(source)
	{
	case KI7001_TRIG_ON_IMMEDIATE:
		szSource="IMM; ";
		break;
	case KI7001_TRIG_ON_MANUAL:
		szSource="MAN; ";
		break;
	case KI7001_TRIG_ON_BUS:
		szSource="BUS; ";
		break;
	case KI7001_TRIG_ON_EXTERNAL:
		szSource="EXT; ";
		break;
	case KI7001_TRIG_ON_TLINK:
		szSource="TLIN; ";
		break;
	default:
		return VI_ERROR_PARAMETER2;
	}
	
	if(direction==KI7001_SOURCE) szDirection="SOUR; ";
	else  szDirection="ACC; ";
	
	if(count==0) Fmt(countbuf,"%s<%s","INF");
	else Fmt(countbuf,"%s<%d",count);
	
	if(source==KI7001_TRIG_ON_TLINK)
		Fmt(buf,"%s<%s%s%s%s%s%s%s%d%s%d",":ARM:SOUR ",szSource,"COUN ",countbuf,"; TCON:DIR ",szDirection,"ASYN:ILIN ",triggerLinkInputLine,"; OLIN ",triggerLinkOutputLine);
	else
		Fmt(buf,"%s<%s%s%s%s%s%s",":ARM:SOUR ",szSource,"COUN ",countbuf,"; TCON:DIR ",szDirection);

    if ((Ki7001_status = viWrite (instrumentHandle, buf, StringLength(buf), &retCnt)) < 0)
        return Ki7001_status;

	if ((Ki7001_status = Ki7001_errorQuery (instrumentHandle, &errorCode, errorMessage)) < 0)
		return Ki7001_status;
	if(errorCode!=0) 
		return Ki7001_ERROR_INVALID_CONFIGURATION;

	return Ki7001_status;					  
}

ViStatus Ki7001_SetupScanLayer (ViSession instrumentHandle, ViInt16 source,
                                ViInt16 count, ViInt16 direction, ViReal64 timer,
                                ViReal64 delay, ViInt16 triggerLinkInputLine,
                                ViInt16 triggerLinkOutputLine)
{
	char buf[256];
	char countbuf[25];
	char *szSource;
	char *szDirection;
    ViUInt32 retCnt = 0;
    ViInt32 errorCode;
    ViChar errorMessage[256];
    ViStatus Ki7001_status = VI_SUCCESS;

	switch(source)
	{
	case KI7001_TRIG_ON_IMMEDIATE:
		szSource="IMM; ";
		break;
	case KI7001_TRIG_ON_MANUAL:
		szSource="MAN; ";
		break;
	case KI7001_TRIG_ON_BUS:
		szSource="BUS; ";
		break;
	case KI7001_TRIG_ON_EXTERNAL:
		szSource="EXT; ";
		break;
	case KI7001_TRIG_ON_TLINK:
		szSource="TLIN; ";
		break;
	case KI7001_TRIG_ON_TIMER:
		szSource="TIM; ";
		break;
	default:
		return VI_ERROR_PARAMETER2;
	}
	
	if(direction==KI7001_SOURCE) szDirection="SOUR; ";
	else  szDirection="ACC; ";
	
	if(count==0) Fmt(countbuf,"%s<%s","INF");
	else Fmt(countbuf,"%s<%d",count);
	
	if(source==KI7001_TRIG_ON_TLINK)
		Fmt(buf,"%s<%s%s%s%s%s%f%s%f%s%s%s%d%s%d",":ARM:LAY2:SOUR ",szSource,"COUN ",countbuf,"; DEL ",delay,"; TIM ",timer,"; TCON:DIR ",szDirection,"ASYN:ILIN ",triggerLinkInputLine,"; OLIN ",triggerLinkOutputLine);
	else
		Fmt(buf,"%s<%s%s%s%s%s%f%s%f%s%s",":ARM:LAY2:SOUR ",szSource,"COUN ",countbuf,"; DEL ",delay,"; TIM ",timer,"; TCON:DIR ",szDirection);

    if ((Ki7001_status = viWrite (instrumentHandle, buf, StringLength(buf), &retCnt)) < 0)
        return Ki7001_status;

	if ((Ki7001_status = Ki7001_errorQuery (instrumentHandle, &errorCode, errorMessage)) < 0)
		return Ki7001_status;
	if(errorCode!=0) 
		return Ki7001_ERROR_INVALID_CONFIGURATION;

	return Ki7001_status;					  
}

ViStatus Ki7001_SetupChanLayer (ViSession instrumentHandle, ViInt16 source,
                                ViInt16 count, ViInt16 direction, ViReal64 timer,
                                ViReal64 delay, ViInt16 triggerLinkInputLine,
                                ViInt16 triggerLinkOutputLine)
{
	char buf[256];
	char countbuf[256];
	char *szSource;
	char *szDirection;
    ViUInt32 retCnt = 0;
    ViInt32 errorCode;
    ViChar errorMessage[256];
    ViStatus Ki7001_status = VI_SUCCESS;

	switch(source)
	{
	case KI7001_TRIG_ON_IMMEDIATE:
		szSource="IMM; ";
		break;
	case KI7001_TRIG_ON_MANUAL:
		szSource="MAN; ";
		break;
	case KI7001_TRIG_ON_BUS:
		szSource="BUS; ";
		break;
	case KI7001_TRIG_ON_EXTERNAL:
		szSource="EXT; ";
		break;
	case KI7001_TRIG_ON_TLINK:
		szSource="TLIN; ";
		break;
	case KI7001_TRIG_ON_TIMER:
		szSource="TIM; ";
		break;
	default:
		return VI_ERROR_PARAMETER2;
	}
	
	if(direction==KI7001_SOURCE) szDirection="SOUR; ";
	else  szDirection="ACC; ";
															
	if(source==KI7001_TRIG_ON_TLINK)
		Fmt(buf,"%s<%s%s%s%f%s%f%s%s%s%d%s%d",":TRIG:SOUR ",szSource,"; DEL ",delay,"; TIM ",timer,"; TCON:DIR ",szDirection,"ASYN:ILIN ",triggerLinkInputLine,"; OLIN ",triggerLinkOutputLine);
	else
		Fmt(buf,"%s<%s%s%s%f%s%f%s%s",":TRIG:SOUR ",szSource,"; DEL ",delay,"; TIM ",timer,"; TCON:DIR ",szDirection);

    if ((Ki7001_status = viWrite (instrumentHandle, buf, StringLength(buf), &retCnt)) < 0)
        return Ki7001_status;

	if ((Ki7001_status = Ki7001_errorQuery (instrumentHandle, &errorCode, errorMessage)) < 0)
		return Ki7001_status;
	if(errorCode!=0) 
		return Ki7001_ERROR_INVALID_CONFIGURATION;

	if(count==0) Fmt(countbuf,"%s<%s"," TRIG:COUNT INF; :TRIG:COUNT:AUTO OFF;");
	else 
	{
		if(count==-1)
			Fmt(countbuf,"%s<%s",":TRIG:COUNT:AUTO ON;");
		else
			Fmt(countbuf,"%s<%s%d",":TRIG:COUNT ",count,"; :TRIG:COUNT:AUTO OFF");
	}
	
    if ((Ki7001_status = viWrite (instrumentHandle, countbuf, StringLength(countbuf), &retCnt)) < 0)
        return Ki7001_status;

	if ((Ki7001_status = Ki7001_errorQuery (instrumentHandle, &errorCode, errorMessage)) < 0)
		return Ki7001_status;
	if(errorCode!=0) 
		return Ki7001_ERROR_INVALID_CONFIGURATION;

	return Ki7001_status;					  
}

ViStatus Ki7001_Bypass (ViSession instrumentHandle, ViInt16 Layer, int bypassDelay)
{
	char buf[256];
	char *cmd;
    ViUInt32 retCnt = 0;
    ViInt32 errorCode;
    ViChar errorMessage[256];
    ViStatus Ki7001_status = VI_SUCCESS;

	if(bypassDelay==1)
		cmd="IMM;";
	else
		cmd="SIGN;";

	switch(Layer)
	{
	case KI7001_ARM_LAYER:
		Fmt(buf,"%s<%s%s",":ARM:",cmd);
		break;
	case KI7001_SCAN_LAYER:
		Fmt(buf,"%s<%s",":ARM:LAY2:",cmd);
		break;
	case KI7001_CHAN_LAYER:
		Fmt(buf,"%s<%s",":TRIG:",cmd);
		break;
	default:
		return VI_ERROR_PARAMETER2;
	}
	
    if ((Ki7001_status = viWrite (instrumentHandle, buf, StringLength(buf), &retCnt)) < 0)
        return Ki7001_status;

	if ((Ki7001_status = Ki7001_errorQuery (instrumentHandle, &errorCode, errorMessage)) < 0)
		return Ki7001_status;
	if(errorCode!=0) 
		return Ki7001_ERROR_INVALID_CONFIGURATION;

	return Ki7001_status;					  
}

ViStatus Ki7001_Start (ViSession instrumentHandle, int continuousInit)
{
	char *cmd;
    ViUInt32 retCnt = 0;
    ViInt32 errorCode;
    ViChar errorMessage[256];
    ViStatus Ki7001_status = VI_SUCCESS;

	if(continuousInit==1)
		cmd=":INIT:CONT ON;";
	else
		cmd=":INIT";

	
    if ((Ki7001_status = viWrite (instrumentHandle, cmd, StringLength(cmd), &retCnt)) < 0)
        return Ki7001_status;

	if ((Ki7001_status = Ki7001_errorQuery (instrumentHandle, &errorCode, errorMessage)) < 0)
		return Ki7001_status;
	if(errorCode!=0) 
		return Ki7001_ERROR_INVALID_CONFIGURATION;

	return Ki7001_status;					  
}

ViStatus Ki7001_Stop (ViSession instrumentHandle)
{
	char *cmd;
    ViUInt32 retCnt = 0;
    ViInt32 errorCode;
    ViChar errorMessage[256];
    ViStatus Ki7001_status = VI_SUCCESS;

	cmd=":INIT:CONT OFF; :ABORT;";
	
    if ((Ki7001_status = viWrite (instrumentHandle, cmd, StringLength(cmd), &retCnt)) < 0)
        return Ki7001_status;

	if ((Ki7001_status = Ki7001_errorQuery (instrumentHandle, &errorCode, errorMessage)) < 0)
		return Ki7001_status;
	if(errorCode!=0) 
		return Ki7001_ERROR_INVALID_CONFIGURATION;

	return Ki7001_status;					  
}

ViStatus Ki7001_DefineScannerCard (ViSession instrumentHandle, ViInt16 slot,
                                   ViInt16 type)
{
	char buf[256];
    ViUInt32 retCnt = 0;
    ViInt32 errorCode;
    ViChar errorMessage[256];
    ViStatus Ki7001_status = VI_SUCCESS;
    
	Fmt(buf,"%s<%s%d%s%d",":ROUT:CONF:SLOT",slot,":CTYPE C",type);
    
    if ((Ki7001_status = viWrite (instrumentHandle, buf, StringLength(buf), &retCnt)) < 0)
        return Ki7001_status;

	if ((Ki7001_status = Ki7001_errorQuery (instrumentHandle, &errorCode, errorMessage)) < 0)
		return Ki7001_status;

	if(errorCode!=0) 
		return Ki7001_ERROR_INVALID_CONFIGURATION;
	
	return Ki7001_status;
}

ViStatus Ki7001_QueryScannerCard (ViSession instrumentHandle, ViInt16 slot,
                                  ViInt16 *type)
{
	char buf[256];
    ViUInt32 retCnt = 0;
    ViInt32 errorCode;
    ViChar errorMessage[256];
    ViChar card[64];
    ViChar *pCard;
    ViStatus Ki7001_status = VI_SUCCESS;
    int t;
    
	Fmt(buf,"%s<%s%d%s",":ROUT:CONF:SLOT",slot,":CTYPE?");
    
    if ((Ki7001_status = viWrite (instrumentHandle, buf, StringLength(buf), &retCnt)) < 0)
        return Ki7001_status;

    if ((Ki7001_status = viScanf (instrumentHandle, "%s", card)) < 0)
        return Ki7001_status;
        
    pCard=&card[1];
    Fmt(&t,"%d<%s",pCard);
	*type=t;
			
	if ((Ki7001_status = Ki7001_errorQuery (instrumentHandle, &errorCode, errorMessage)) < 0)
		return Ki7001_status;

	if(errorCode!=0) 
		return Ki7001_ERROR_INVALID_CONFIGURATION;
	
	return Ki7001_status;
}

ViStatus Ki7001_SetPoles (ViSession instrumentHandle, ViInt16 slot, ViInt16 poles)
{
	char buf[256];
    ViUInt32 retCnt = 0;
    ViInt32 errorCode;
    ViChar errorMessage[256];
    ViStatus Ki7001_status = VI_SUCCESS;
    
	Fmt(buf,"%s<%s%d%s%d",":ROUT:CONF:SLOT",slot,":POLE ",poles);
    
    if ((Ki7001_status = viWrite (instrumentHandle, buf, StringLength(buf), &retCnt)) < 0)
        return Ki7001_status;

	if ((Ki7001_status = Ki7001_errorQuery (instrumentHandle, &errorCode, errorMessage)) < 0)
		return Ki7001_status;

	if(errorCode!=0) 
		return Ki7001_ERROR_INVALID_CONFIGURATION;
	
	return Ki7001_status;
}

ViStatus Ki7001_QueryPoles (ViSession instrumentHandle, ViInt16 slot,
                            ViInt16 *poles)
{
	char buf[256];
    ViUInt32 retCnt = 0;
    ViInt32 errorCode;
    ViChar errorMessage[256];
    ViChar reply[64];
    ViStatus Ki7001_status = VI_SUCCESS;
    int t;
    
	Fmt(buf,"%s<%s%d%s",":ROUT:CONF:SLOT",slot,":POLE?");
    
    if ((Ki7001_status = viWrite (instrumentHandle, buf, StringLength(buf), &retCnt)) < 0)
        return Ki7001_status;

    if ((Ki7001_status = viScanf (instrumentHandle, "%s", reply)) < 0)
        return Ki7001_status;
        
    Fmt(&t,"%d<%s",reply);
	*poles=t;
			
	if ((Ki7001_status = Ki7001_errorQuery (instrumentHandle, &errorCode, errorMessage)) < 0)
		return Ki7001_status;

	if(errorCode!=0) 
		return Ki7001_ERROR_INVALID_CONFIGURATION;
	
	return Ki7001_status;
}

ViStatus Ki7001_SetSettlingTime (ViSession instrumentHandle, ViInt16 slot,
                                 ViReal64 settlingTime)
{
	char buf[256];
    ViUInt32 retCnt = 0;
    ViInt32 errorCode;
    ViChar errorMessage[256];
    ViStatus Ki7001_status = VI_SUCCESS;
    
	Fmt(buf,"%s<%s%d%s%f",":ROUT:CONF:SLOT",slot,":STIME ",settlingTime);
    
    if ((Ki7001_status = viWrite (instrumentHandle, buf, StringLength(buf), &retCnt)) < 0)
        return Ki7001_status;

	if ((Ki7001_status = Ki7001_errorQuery (instrumentHandle, &errorCode, errorMessage)) < 0)
		return Ki7001_status;

	if(errorCode!=0) 
		return Ki7001_ERROR_INVALID_CONFIGURATION;
	
	return Ki7001_status;
}

ViStatus Ki7001_QuerySettlingTime (ViSession instrumentHandle, ViInt16 slot,
                                   ViReal64 *settlingTime)
{
	char buf[256];
    ViUInt32 retCnt = 0;
    ViInt32 errorCode;
    ViChar errorMessage[256];
    ViChar reply[64];
    ViStatus Ki7001_status = VI_SUCCESS;
    double t;
    
	Fmt(buf,"%s<%s%d%s",":ROUT:CONF:SLOT",slot,":STIME?");
    
    if ((Ki7001_status = viWrite (instrumentHandle, buf, StringLength(buf), &retCnt)) < 0)
        return Ki7001_status;

    if ((Ki7001_status = viScanf (instrumentHandle, "%s", reply)) < 0)
        return Ki7001_status;
        
    Fmt(&t,"%f<%s",reply);
	*settlingTime=t;
			
	if ((Ki7001_status = Ki7001_errorQuery (instrumentHandle, &errorCode, errorMessage)) < 0)
		return Ki7001_status;

	if(errorCode!=0) 
		return Ki7001_ERROR_INVALID_CONFIGURATION;
	
	return Ki7001_status;
}

ViStatus Ki7001_EnableBBM (ViSession instrumentHandle, ViInt16 enable)
{
	char buf[256];
    ViUInt32 retCnt = 0;
    ViInt32 errorCode;
    ViChar errorMessage[256];
    ViStatus Ki7001_status = VI_SUCCESS;
    
	Fmt(buf,"%s<%s%d",":ROUT:CONF:BBM ",enable);
    
    if ((Ki7001_status = viWrite (instrumentHandle, buf, StringLength(buf), &retCnt)) < 0)
        return Ki7001_status;

	if ((Ki7001_status = Ki7001_errorQuery (instrumentHandle, &errorCode, errorMessage)) < 0)
		return Ki7001_status;

	if(errorCode!=0) 
		return Ki7001_ERROR_INVALID_CONFIGURATION;
	
	return Ki7001_status;
}

ViStatus Ki7001_QueryBBM (ViSession instrumentHandle, ViInt16 *BBM)
{
	char buf[256];
    ViUInt32 retCnt = 0;
    ViInt32 errorCode;
    ViChar errorMessage[256];
    ViChar reply[64];
    ViStatus Ki7001_status = VI_SUCCESS;
    int t;
    
	Fmt(buf,"%s<%s",":ROUT:CONF:BBM?");
    
    if ((Ki7001_status = viWrite (instrumentHandle, buf, StringLength(buf), &retCnt)) < 0)
        return Ki7001_status;

    if ((Ki7001_status = viScanf (instrumentHandle, "%s", reply)) < 0)
        return Ki7001_status;
        
    Fmt(&t,"%d<%s",reply);
	*BBM=t;
			
	if ((Ki7001_status = Ki7001_errorQuery (instrumentHandle, &errorCode, errorMessage)) < 0)
		return Ki7001_status;

	if(errorCode!=0) 
		return Ki7001_ERROR_INVALID_CONFIGURATION;
	
	return Ki7001_status;
}

ViStatus Ki7001_EnableSingleChan (ViSession instrumentHandle, ViInt16 enable)
{
	char buf[256];
    ViUInt32 retCnt = 0;
    ViInt32 errorCode;
    ViChar errorMessage[256];
    ViStatus Ki7001_status = VI_SUCCESS;
    
	Fmt(buf,"%s<%s%d",":ROUT:CONF:SCH ",enable);
    
    if ((Ki7001_status = viWrite (instrumentHandle, buf, StringLength(buf), &retCnt)) < 0)
        return Ki7001_status;

	if ((Ki7001_status = Ki7001_errorQuery (instrumentHandle, &errorCode, errorMessage)) < 0)
		return Ki7001_status;

	if(errorCode!=0) 
		return Ki7001_ERROR_INVALID_CONFIGURATION;
	
	return Ki7001_status;
}

ViStatus Ki7001_QuerySingleChan (ViSession instrumentHandle, ViInt16 *mode)
{
	char buf[256];
    ViUInt32 retCnt = 0;
    ViInt32 errorCode;
    ViChar errorMessage[256];
    ViChar reply[64];
    ViStatus Ki7001_status = VI_SUCCESS;
    int t;
    
	Fmt(buf,"%s<%s",":ROUT:CONF:SCH?");
    
    if ((Ki7001_status = viWrite (instrumentHandle, buf, StringLength(buf), &retCnt)) < 0)
        return Ki7001_status;

    if ((Ki7001_status = viScanf (instrumentHandle, "%s", reply)) < 0)
        return Ki7001_status;
        
    Fmt(&t,"%d<%s",reply);
	*mode=t;
			
	if ((Ki7001_status = Ki7001_errorQuery (instrumentHandle, &errorCode, errorMessage)) < 0)
		return Ki7001_status;

	if(errorCode!=0) 
		return Ki7001_ERROR_INVALID_CONFIGURATION;
	
	return Ki7001_status;
}

ViStatus Ki7001_EnableCardPairing (ViSession instrumentHandle, ViInt16 enable)
{
	char buf[256];
    ViUInt32 retCnt = 0;
    ViInt32 errorCode;
    ViChar errorMessage[256];
    ViStatus Ki7001_status = VI_SUCCESS;
    
	Fmt(buf,"%s<%s%d",":ROUT:CONF:CPA ",enable);
    
    if ((Ki7001_status = viWrite (instrumentHandle, buf, StringLength(buf), &retCnt)) < 0)
        return Ki7001_status;

	if ((Ki7001_status = Ki7001_errorQuery (instrumentHandle, &errorCode, errorMessage)) < 0)
		return Ki7001_status;

	if(errorCode!=0) 
		return Ki7001_ERROR_INVALID_CONFIGURATION;
	
	return Ki7001_status;
}

ViStatus Ki7001_QueryCardPairing (ViSession instrumentHandle, ViInt16 *mode)
{
	char buf[256];
    ViUInt32 retCnt = 0;
    ViInt32 errorCode;
    ViChar errorMessage[256];
    ViChar reply[64];
    ViStatus Ki7001_status = VI_SUCCESS;
    int t;
    
	Fmt(buf,"%s<%s",":ROUT:CONF:CPA?");
    
    if ((Ki7001_status = viWrite (instrumentHandle, buf, StringLength(buf), &retCnt)) < 0)
        return Ki7001_status;

    if ((Ki7001_status = viScanf (instrumentHandle, "%s", reply)) < 0)
        return Ki7001_status;
        
    Fmt(&t,"%d<%s",reply);
	*mode=t;
			
	if ((Ki7001_status = Ki7001_errorQuery (instrumentHandle, &errorCode, errorMessage)) < 0)
		return Ki7001_status;

	if(errorCode!=0) 
		return Ki7001_ERROR_INVALID_CONFIGURATION;
	
	return Ki7001_status;
}

char *Ki7001_MAKESCANCHAN (char *buffer, int slot, int row, int column)
{
	Fmt(buffer,"%s<%d%s%d%s%d",slot,"!",row,"!",column);
	return buffer;
}

char *Ki7001_MAKEMUXCHAN (char *buffer, int slot, int channel)
{
	Fmt(buffer,"%s<%d%s%d",slot,"!",channel);
	return buffer;
}

char *Ki7001_MAKEMEMCHAN (char *buffer, int memory)
{
	Fmt(buffer,"%s<%s%d","M",memory);
	return buffer;
}

char *Ki7001_MAKECONTIGUOUSLIST (char *buffer, char *start, char *stop)
{
	Fmt(buffer,"%s<%s%s%s",start,":",stop);
	return buffer;
}

char *Ki7001_ADDTOLIST (char *buffer, char *actual, char *new)
{
	Fmt(buffer,"%s<%s%s%s",actual,",",new);
	return buffer;
}

ViStatus Ki7001_Close (ViSession instrumentHandle, ViChar channelList[])
{
	char buf[4096];
    ViUInt32 retCnt = 0;
    ViInt32 errorCode;
    ViChar errorMessage[256];
    ViStatus Ki7001_status = VI_SUCCESS;
    
	Fmt(buf,"%s<%s%s%s",":ROUT:CLOS (@ ",channelList,")");
    
    if ((Ki7001_status = viWrite (instrumentHandle, buf, StringLength(buf), &retCnt)) < 0)
        return Ki7001_status;

	if ((Ki7001_status = Ki7001_errorQuery (instrumentHandle, &errorCode, errorMessage)) < 0)
		return Ki7001_status;

	if(errorCode!=0) 
		return Ki7001_ERROR_INVALID_CONFIGURATION;
	
	return Ki7001_status;
}

ViStatus Ki7001_Open (ViSession instrumentHandle, ViChar channelList[])
{
	char buf[4096];
    ViUInt32 retCnt = 0;
    ViInt32 errorCode;
    ViChar errorMessage[256];
    ViStatus Ki7001_status = VI_SUCCESS;
    
	Fmt(buf,"%s<%s%s%s",":ROUT:OPEN (@ ",channelList,")");
    
    if ((Ki7001_status = viWrite (instrumentHandle, buf, StringLength(buf), &retCnt)) < 0)
        return Ki7001_status;

	if ((Ki7001_status = Ki7001_errorQuery (instrumentHandle, &errorCode, errorMessage)) < 0)
		return Ki7001_status;

	if(errorCode!=0) 
		return Ki7001_ERROR_INVALID_CONFIGURATION;
	
	return Ki7001_status;
}

ViStatus Ki7001_OpenAll (ViSession instrumentHandle)
{
	char buf[4096];
    ViUInt32 retCnt = 0;
    ViInt32 errorCode;
    ViChar errorMessage[256];
    ViStatus Ki7001_status = VI_SUCCESS;
    
	Fmt(buf,"%s<%s",":ROUT:OPEN ALL");
    
    if ((Ki7001_status = viWrite (instrumentHandle, buf, StringLength(buf), &retCnt)) < 0)
        return Ki7001_status;

	if ((Ki7001_status = Ki7001_errorQuery (instrumentHandle, &errorCode, errorMessage)) < 0)
		return Ki7001_status;

	if(errorCode!=0) 
		return Ki7001_ERROR_INVALID_CONFIGURATION;
	
	return Ki7001_status;
}

ViStatus Ki7001_ScanList (ViSession instrumentHandle, ViChar channelList[])
{
	char buf[4096];
    ViUInt32 retCnt = 0;
    ViInt32 errorCode;
    ViChar errorMessage[256];
    ViStatus Ki7001_status = VI_SUCCESS;
    
	Fmt(buf,"%s<%s%s%s",":ROUT:SCAN (@ ",channelList,")");
    
    if ((Ki7001_status = viWrite (instrumentHandle, buf, StringLength(buf), &retCnt)) < 0)
        return Ki7001_status;

	if ((Ki7001_status = Ki7001_errorQuery (instrumentHandle, &errorCode, errorMessage)) < 0)
		return Ki7001_status;

	if(errorCode!=0) 
		return Ki7001_ERROR_INVALID_CONFIGURATION;
	
	return Ki7001_status;
}

ViStatus Ki7001_StoreRelayPattern (ViSession instrumentHandle, ViInt16 pattern)
{
	char buf[256];
    ViUInt32 retCnt = 0;
    ViInt32 errorCode;
    ViChar errorMessage[256];
    ViStatus Ki7001_status = VI_SUCCESS;
    
	Fmt(buf,"%s<%s%d",":ROUT:MEM:SAVE M",pattern);
    
    if ((Ki7001_status = viWrite (instrumentHandle, buf, StringLength(buf), &retCnt)) < 0)
        return Ki7001_status;

	if ((Ki7001_status = Ki7001_errorQuery (instrumentHandle, &errorCode, errorMessage)) < 0)
		return Ki7001_status;

	if(errorCode!=0) 
		return Ki7001_ERROR_INVALID_CONFIGURATION;
	
	return Ki7001_status;
}

ViStatus Ki7001_RecallRelayPattern (ViSession instrumentHandle, ViInt16 pattern)
{
	char buf[256];
    ViUInt32 retCnt = 0;
    ViInt32 errorCode;
    ViChar errorMessage[256];
    ViStatus Ki7001_status = VI_SUCCESS;
    
	Fmt(buf,"%s<%s%d",":ROUT:MEM:REC M",pattern);
    
    if ((Ki7001_status = viWrite (instrumentHandle, buf, StringLength(buf), &retCnt)) < 0)
        return Ki7001_status;

	if ((Ki7001_status = Ki7001_errorQuery (instrumentHandle, &errorCode, errorMessage)) < 0)
		return Ki7001_status;

	if(errorCode!=0) 
		return Ki7001_ERROR_INVALID_CONFIGURATION;
	
	return Ki7001_status;
}

/*****************************************************************************/
/*=== END INSTRUMENT DRIVER SOURCE CODE =====================================*/
/*****************************************************************************/

